const path = require('path');
const express = require('express');
const cors = require('cors');
const http = require('http');
const { Server } = require('socket.io');
const pagesRouter = require('./routes/pages');
const orderRouter = require('./routes/order');

const app = express();
app.use(cors());
app.use(express.json());

// Serve static front-end (login page, etc.)
app.use(express.static(path.join(__dirname, 'public')));

// API routes
app.use('/api/pages', pagesRouter);
app.use('/api/order', orderRouter);

// Optional: direct route to login page
app.get('/login', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'login.html'));
});

// HTTP + Socket.io
const server = http.createServer(app);
const io = new Server(server, { cors: { origin: "*" } });

// Shared state for live page navigation
let pagesOrder = ['view', 'dashboard', 'update'];
let currentPageIndex = 0;

io.on('connection', (socket) => {
  // Send current page on connect
  socket.emit('pageChange', pagesOrder[currentPageIndex]);

  socket.on('nextPage', () => {
    if (currentPageIndex < pagesOrder.length - 1) currentPageIndex++;
    io.emit('pageChange', pagesOrder[currentPageIndex]);
  });

  socket.on('prevPage', () => {
    if (currentPageIndex > 0) currentPageIndex--;
    io.emit('pageChange', pagesOrder[currentPageIndex]);
  });

  socket.on('setOrder', (order) => {
    if (Array.isArray(order) && order.length) {
      pagesOrder = order;
      currentPageIndex = 0;
      io.emit('pageChange', pagesOrder[currentPageIndex]);
    }
  });
});

const PORT = process.env.PORT || 4000;
server.listen(PORT, () => {
  console.log(`Backend running on http://localhost:${PORT}`);
});
